% This function calculates the terms in Eq.13, 16 & 23.

% Sc (terms in Eq.13, 16 & 23):
% 1)psi; 2)E1; 3)E2; 4)E(Tr); 5)Xi; 6)E3; 7)E4; 8)E(Tn); 9)E(Tq).

% Wat: catchment- and storm-average runoff coefficient;
% Rat: catchment- and storm-average rainfall excess;
% Ra: catchment-average rainfall excess;
% Rt: storm-average rainfall excess;
% alpha: a velocity coefficient used to distinguish the routing time for 
%        different rainfall excess components;
% ThetaI/ThetaO: subsurface/surface routing time field;

function Sc=AFC(Wat,Rat,Ra,Rt,alpha,ThetaI,ThetaO)

Area=find(~isnan(ThetaI));

% Equation 12
if Rat(3)~=0
  psi=[Rat(1)/Rat(3) Rat(2)/Rat(3)];
else
  psi=[Wat(1)/Wat(3) Wat(2)/Wat(3)];
end

% Equation 16 (14)
E1=[(size(Ra,1)+1)/2 (size(Ra,1)+1)/2 (size(Ra,1)+1)/2];

cc=cov(1:size(Ra,1),Ra(:,1)/Rat(1),1);
EI2=cc(2,1);
cc=cov(1:size(Ra,1),Ra(:,2)/Rat(2),1);
EO2=cc(2,1);

E2=[EI2 EO2 psi(1)*EI2+psi(2)*EO2];

ETr=nansum([E1;E2],1);

% Equation 21
Xi=[psi(1)+alpha*psi(2) psi(1)/alpha+psi(2)];

% Equation 23 (17) E3
E3=[mean(ThetaI(Area)) mean(ThetaO(Area))...
    Xi(1)*psi(1)*mean(ThetaI(Area))+Xi(2)*psi(2)*mean(ThetaO(Area))];

rtI=Rt(:,:,1);
cc=cov(ThetaI(Area),rtI(Area)/Rat(1),1);
EI4=cc(2,1);
rtO=Rt(:,:,2);
cc=cov(ThetaO(Area),rtO(Area)/Rat(2),1);
EO4=cc(2,1);

E4=[EI4 EO4 Xi(1)*psi(1)*EI4+Xi(2)*psi(2)*EO4];

ETn=nansum([E3;E4],1);

% Equation 13
ETq=ETr+ETn;

% Output (9 by 3)
Sc=[[psi 1];E1;E2;ETr;[Xi sum(Xi)];E3;E4;ETn;ETq];
end
